const express = require("express");
const connectDB = require("./config/config");
// Import the Book model to interact with the 'books' collection in the database.
const Book = require("./models/Book");

// --- 2. INITIALIZATION ---
// Create an instance of the Express application.
const app = express();
// Connect to the MongoDB database.
connectDB();

// --- 3. MIDDLEWARE ---
// Use the express.json() middleware to parse incoming JSON request bodies.
// This is necessary to access data sent in POST, PUT, or PATCH requests (e.g., req.body).
app.use(express.json());

// --- 4. ROUTES ---
app.post("/api/books", async (req, res) => {
  try {
    const { title, author, year, genre } = req.body;

    const newBook = new Book({ title, author, year, genre });
    await newBook.save();
  } catch (err) {
    res.status(500).json({ error: "Erro ao criar livro" });
  }
});
// A simple test route from your class to demonstrate try-catch.
app.get("/api/books", async (req, res) => {
  try {
    const books = await Book.find();
    res.json(books);
  } catch (err) {
    res.status(500).json({ error: "erro ao buscar livros"})
  }
})

/**
 * POST /books
 * Route to create a new book.
 * It expects a JSON body with 'title', 'author', and optionally 'year'.
 */
app.post('/books', async (req, res) => {
  try {
    // Destructure title, author, and year from the request body.
    const { title, author, year } = req.body;

    // Basic validation: Check if title and author are provided.
    // Mongoose's 'required' schema option also handles this, but this is a good extra check.
    if (!title || !author) {
      throw new Error('Title and author are required fields.');
    }

    // Create a new book document using the Book model.
    const newBook = new Book({ title, author, year });
    
    // Asynchronously save the new book document to the database.
    await newBook.save();

    // Respond with a 201 (Created) status and the newly created book data.
    res.status(201).json(newBook);

  } catch (err) {
    // If any error occurs (validation, database error, etc.), catch it here.
    // Respond with a 400 (Bad Request) status and the error message.
    res.status(400).json({ error: err.message });
  }
});

/**
 * GET /books
 * Route to retrieve all books from the database.
 */
app.get('/books', async (req, res) => {
  try {
    // Use the Book model's find() method to get all documents from the 'books' collection.
    const books = await Book.find();
    // Respond with a 200 (OK) status and the array of books.
    res.status(200).json(books);
  } catch (err) {
    // If a database error occurs, send a 500 (Internal Server Error) status.
    res.status(500).json({ error: "An error occurred while fetching books." });
  }
});


// --- 5. SERVER START ---
// Define the port the server will listen on.
const PORT = 3000;
// Start the server and log a message to the console once it's running.
app.listen(PORT, () => console.log(`Server running on port ${PORT}`));